## ----init,include=FALSE--------------------------------------------------
#set global chunk options
#opts_chunk[["set"]](warning=FALSE, error=FALSE, message=FALSE)
opts_chunk[["set"]](fig.align='center', dev='tikz', external=FALSE, fig.width=5, fig.height=3, fig.show='hold', echo=FALSE,warning=FALSE, error=FALSE, message=FALSE, tidy=FALSE,cache=TRUE,autodep=TRUE, par=TRUE, comment=NA, keep.blank.line=FALSE)
 knit_hooks[["set"]](par=function(before, options, envir){
 if (before) { options(max.print=40) ; options(width=80) }
 if (before && options[["fig.show"]]!='none') par(mar=c(4,4,0,.1),cex.main=1,font.main=1,cex.lab=.95,cex.axis=.9,mgp=c(2,.7,0),tcl=-.3)
 }, crop=hook_pdfcrop)

## ----setup, include=FALSE, cache=FALSE-----------------------------------
library(car)
library(pwr)
library(coin) ## for wilcox_test
library(fmsb) ## for risk_ratio
library(lattice)
library(latticeExtra)
library(lme4) 
library(memisc)
library(plyr)
library(runjags)
library(coda)
library(vcd)
library(xtable)
library(zTree)
library(magrittr)
#
ltheme <- canonical.theme(color = FALSE)
ltheme[["strip.background"]][["col"]] <- "transparent"
ltheme[["par.main.text"]][["font"]]=1
ltheme[["par.main.text"]][["cex"]]=1
ltheme[["par.sub.text"]][["font"]]=1
lattice.options(default.theme = ltheme)
#
mPal<-brewer.pal(6,"Greys")
mTheme<-custom.theme(fill = mPal)
mTheme[["par.main.text"]][["font"]]=1
mTheme[["par.main.text"]][["cex"]]=1
mTheme[["par.sub.text"]][["font"]]=1
#
options(xtable.floating=FALSE)
options(xtable.sanitize.text.function=function(x) x)
options(xtable.include.rownames=FALSE)

## ----testing-------------------------------------------------------------
treatSort<-c("MDT","CDT","SDT")
cTrans<-read.csv(text="var,tex
computer,CDT
multiple,MDT
single,SDT")
treatTrans<-function(n) {
    for(i in 1:nrow(cTrans)) n<-sub(cTrans[i,1],cTrans[i,2],n)
    n
}
pval<-function(p) sprintf("%.4f",p)
infTex<-function(x) gsub("Inf","\\\\infty",x)
##
Tt.test<-function(x,t,alternative="two.sided",group="treatmenttype",CI=FALSE) {
    xx<-subset(GesamtDaten,treatmenttype!=t)
    testRes<-t.test(xx[,x] ~ xx[,group],alternative=alternative)
    if(!CI) return(pval(testRes[["p.value"]]))
    infTex(sprintf("$\\Delta$=%.4g \\begin{tabular}{c}CI=$[%.4g,%.4g]$\\\\($p=%.4f$)\\end{tabular}",
            testRes[["estimate"]] %*% c(1,-1),
            testRes[["conf.int"]][1],
            testRes[["conf.int"]][2],
            testRes[["p.value"]]))
}
##
pwr.test <- function (x,rows=c("\\dictator","\\responder"),t,alternative="two.sided",group="treatmenttype",CI=TRUE) {
    x<-intersect(paste(x,c("","_r"),sep=""),names(GesamtDaten))
    xx<-GesamtDaten[,c(x,group)]
    if(ncol(xx)==3) {
        names(xx)<-c("y.1","y.2","group")
        xx<-reshape(xx,direction="long",varying=c("y.1","y.2"),timevar="role")
    }
    else 
        names(xx) <- c("y","group")
    xx[["group"]]<-factor(xx[["group"]],levels=c("single","computer","multiple"))
    xx<-subset(xx,group!=t & !is.na(y))
    xx.stats<-Aggregate(c(n=length(y),var=var(y)) ~ group,data=xx)
    cname<-paste(treatTrans(xx.stats$group),collapse="--")
    xx.sigma<-with(xx.stats,sqrt(sum((n-1)*var)/sum(n-1)))
    
    powers<-c(.7,.8,.9)
    alphas<-c(.01,.05,.1)
    effSize<-sapply(alphas,function(alpha) 
        sapply(powers,
               function(po) pwr.t2n.test(n1=xx.stats$n[1],n2=xx.stats$n[2],sig.level=alpha,power=po,alternative=alternative)[["d"]]*xx.sigma))
    colnames(effSize)<-sprintf("$\\begin{array}{c}\\alpha=\\\\%g\\end{array}$",alphas)
    rownames(effSize)<-sprintf("$p=%g$",powers)
    xx<-print(xtable(effSize),file="null",include.rownames=TRUE)
    names(xx)<-cname
    xx
}
##
dr.test <- function (x,rows=c("\\dictator","\\responder"),t,alternative="two.sided",group="treatmenttype",CI=TRUE) {
    x<-intersect(paste(x,c("","_r"),sep=""),names(GesamtDaten))
    xx<-GesamtDaten[,c(x,group)]
    if(ncol(xx)==3) {
        names(xx)<-c("y.1","y.2","group")
        xx<-reshape(xx,direction="long",varying=c("y.1","y.2"),timevar="role")
    }
    else 
        names(xx) <- c("y","group")
    xx[["group"]]<-factor(xx[["group"]],levels=c("single","computer","multiple"))
    testRes<-with(subset(xx,group!=t),
                  t.test(y ~ group,alternative=alternative))
    cname<-treatTrans(paste(sub("mean in group ","",attributes(testRes[["estimate"]])[["names"]]),collapse="--"))
    out<-matrix(infTex(sprintf("$\\Delta=%.4g$ \\begin{tabular}{c}CI=$[%.4g,%.4g]$\\\\($p=%.4f$)\\end{tabular}",
            testRes[["estimate"]] %*% c(1,-1),
            testRes[["conf.int"]][1],
            testRes[["conf.int"]][2],
            testRes[["p.value"]])))
    colnames(out)<-cname
    out
    }
##
Gt.test<-function(x,y,t,alternative="two.sided") {
    xx<-subset(GesamtDaten,treatmenttype==t)
    pval(t.test(xx[,x],xx[,y],alternative=alternative)[["p.value"]])
    }
##
mClean <- function(x) {
        within(subset(GesamtDaten,!is.na(GesamtDaten[[x]])),{
               treatmenttype<-factor(treatmenttype)
               AChoice<-ifelse(choiceA,"A","B")
           })[,c("treatmenttype",x,"AChoice")]
}
##
binM.test <- function(x,t,alternative="two.sided") {
    xx<-table(mClean(x)[,c("treatmenttype",x)])
    pval(binom.test(xx[t,c("less","more")],alternative=alternative)[["p.value"]])
}
##
MMosaic <- function(x,ylab="Responsibility") {
    xx<-within(mClean(x),treatmenttype<-treatTrans(treatmenttype))
    if(sum(!is.na(xx[["AChoice"]]))==0) xx[["AChoice"]]<-NULL
    mosaicplot(table(xx),main="",ylab=ylab,xlab="Treatment")
    }
##

## ----ReadingData,error=FALSE,include=FALSE,warning=FALSE,cache=F---------
setwd(".")

# --------------------------------------------------------------------------------#
# ---------------------- Reading the ztree Date into R ---------------------------#
# --------------------------------------------------------------------------------#

#source("http://www.kirchkamp.de/lab/zTree.R")

options(zTree.silent=TRUE) # <- less chatty
setwd("../data")
allFiles<-list.files(".","*.xls",recursive=TRUE)
zTree       <- zTreeTables( allFiles )
subjects <- zTree[["subjects"]] 
 
#um Subject files (also questionnaire zu buchen)
files <- list.files(pattern = "*.sbj$",recursive=TRUE)
#fname <- sub(".*/","",files)
#sbj <- zTreeSbj(aggregate(files,list(fname),function(x) x[1])[["x"]]) 
sbj <- zTreeSbj(files)
setwd("../paper")
sbj<-sbj[order(sbj[["Date"]],as.numeric(sbj[["Subject"]])),]

#generate individual subject ID
subjects<-within(subjects,{sid<-as.character(paste(Date,Subject,sep="_"))})
sbj<-within(sbj,{sid<-as.character(paste(Date,Subject,sep="_"))})

#remove experimentator
subjects<-subjects[subjects[["Role"]]=='1',]
sbj<-sbj[sbj[["client"]]!="local",]

#remove subject 8 on 160713_1530 due to hardware problems
hardwareProblems <- '160713_1530_8';
subjects<-subjects[! subjects[["sid"]] %in% hardwareProblems,]
sbj<-sbj[! sbj[["sid"]] %in% hardwareProblems,]

#save datasets
save(sbj, file = "sbj.RData")
save(subjects, file = "subjects.RData")

#combine all the information into one data frame
GesamtDaten<-merge(subjects,sbj[,grep("Date|Subject",names(sbj),invert=TRUE)
],by="sid")

#save datasets
save(GesamtDaten, file = "GesamtDaten.RData")

#clean up values
rm (allFiles, files, zTree)

# --------------------------------------------------------------------------------#
# --------------------------------------------------------------------------------#
# ---------------------- Alternative if no internet ------------------------------#
# --------------------------------------------------------------------------------#
#load( "subjects.RData")
#load( "sbj.RData")
#load( "GesamtDaten.RData")
#names(GesamtDaten)
#save(subjects,globals,sbj, file = "subjects.RData")
# --------------------------------------------------------------------------------#

## ----DataPreparation,error=FALSE,include=FALSE,warning=FALSE-------------

#generate factor treatmenttype variable [1=multiple dictator treatment (multiple), 2=random mechanism treatment (computer), 3=single dictator treatment (single)]
load( "GesamtDaten.RData")
names(GesamtDaten)
sessionType <- data.frame(t(sapply(by(GesamtDaten,list(GesamtDaten[["Date"]]),function(x) c(single=mean(is.na(x[["choice_Z"]])),multiple=mean(is.na(x[["choice_Y_human"]])))),c)))
sessionType['160713_0933','multiple']<-0
sessionType[["Date"]]<-rownames(sessionType)
sessionType<-within(sessionType,{
    treatmenttype<-NA
    treatmenttype[multiple==0 & single==0]<-"computer"
    treatmenttype[multiple==1]<-"multiple"
    treatmenttype[single==1]<-"single"
    treatmenttype<-factor(treatmenttype)
    })
GesamtDaten<-merge(GesamtDaten,sessionType[,c("treatmenttype","Date")],by="Date")
#
GesamtDaten$treatmenttype
GesamtDaten<-within(GesamtDaten,{
    session<-as.factor(Date)
    choiceA<-NA
    choiceA[GroupRole==1]<-choice_X[GroupRole==1]==1
    choiceA[GroupRole==2 & treatmenttype=="multiple"]<-choice_Y[GroupRole==2 & treatmenttype=="multiple"]==1
    Recipient<-(treatmenttype %in% c("multiple","computer")) & GroupRole==3
    Recipient<-Recipient | (treatmenttype == "single" & GroupRole==2)
    IntermediateRecipient<- (treatmenttype == "computer" & GroupRole==2)
    #expected choices by Recipients
    choiceAExp<-NA
    choiceAExp[Recipient] <- choice_Z[Recipient]
    choiceAExp[Recipient & treatmenttype=="single"] <- choice_Y[Recipient & treatmenttype=="single"]
    choiceAExp[treatmenttype != "multiple"] <- 1 - choiceAExp[treatmenttype != "multiple"]
    #expected choices by IntermediateRecipients ?56 instead of 61 observations as in the first CDT the expectation of the IntermediateRecipients was saved in variable     choice_Y and not in choice_Y_human
    choiceAExp_pd<-NA
    choiceAExp_pd[IntermediateRecipient] <- choice_Y[IntermediateRecipient]
    choiceAExp_pd[IntermediateRecipient] <- choice_Y_human[IntermediateRecipient]
    choiceAExp_pd[IntermediateRecipient] <- 1 - choiceAExp_pd[IntermediateRecipient]
    #
    ownresponsibility<-NA
    ownresponsibility [treatmenttype=="multiple"& GroupRole!=3] <- 100-questionnaire_9XYown[treatmenttype=="multiple"& GroupRole!=3]
    ownresponsibility [treatmenttype=="computer"& GroupRole==1]<-100-questionnaire_9Xown[treatmenttype=="computer"& GroupRole==1]
    ownresponsibility [treatmenttype=="single"& GroupRole==1]<-100-questionnaire_9Xowns[treatmenttype=="single"& GroupRole==1]
    #
    otherresponsibility<-NA
    otherresponsibility [treatmenttype=="multiple" & GroupRole!=3] <- 100-questionnaire_9XYother[treatmenttype=="multiple"& GroupRole!=3]
    otherresponsibility [treatmenttype=="computer" & GroupRole==1] <- 100-questionnaire_9Xcomputer[treatmenttype=="computer"& GroupRole==1]
    #
    diff.ownotherresponsibility <- ownresponsibility - otherresponsibility
    #
    responsBenefPlayer<-NA
    responsBenefPlayer[treatmenttype=="multiple"& GroupRole!=3]<-100-((questionnaire_6X[treatmenttype=="multiple"& GroupRole!=3]+questionnaire_6Y[treatmenttype=="multiple"& GroupRole!=3]))
    responsBenefPlayer[treatmenttype=="computer"& GroupRole==1]<-100-questionnaire_6X[treatmenttype=="computer" & GroupRole==1]
    #
    responsResponder <-NA
    responsResponder [treatmenttype=="multiple"& GroupRole!=3]<-100-(questionnaire_7XY[treatmenttype=="multiple"& GroupRole!=3])
    responsResponder [treatmenttype=="computer"& GroupRole==1]<-100-(questionnaire_7X[treatmenttype=="computer" & GroupRole==1])
    responsResponder [treatmenttype=="single"& GroupRole==1]<-100-(questionnaire_7Xs[treatmenttype=="single" & GroupRole==1])
    #
    guilt<-NA
    guilt[treatmenttype=="multiple"& GroupRole!=3]<-100-questionnaire_8XY[treatmenttype=="multiple"& GroupRole!=3]
    guilt[treatmenttype=="computer"& GroupRole==1]<-100-questionnaire_8X[treatmenttype=="computer" & GroupRole==1]
    guilt[treatmenttype=="single"& GroupRole==1]<-100-questionnaire_8Xs[treatmenttype=="single" & GroupRole==1]
    #
})
save(GesamtDaten, file = "GesamtDaten.RData")
# with(GesamtDaten,table(choiceA))

## ----Descriptives,error=FALSE,include=FALSE,warning=FALSE----------------
#------------------------------------------------number of subjects
#number of total subjects (multiple, computer, single)
nrsubj<-with(GesamtDaten,table(treatmenttype))
#percentage of female subjects (multiple, computer, single)
gender         <-with(GesamtDaten,table(geschlecht,treatmenttype))
propGenderSubj<-round(prop.table(table(GesamtDaten[["geschlecht"]]))*100,1)
propGenderByTreat<-round(apply(gender,2,prop.table)*100,1)
#number of female subjects in percent (multiple, computer, single)

#------------------------------------------------age
#age
meanAge<-round(2016-mean(as.numeric(as.character(GesamtDaten[["geburtsjahr"]])),na.rm=TRUE),0)

#------------------------------------------------payment per treatment
meanpaysum <-round(mean(GesamtDaten[["paysum"]]),2)

## ----diffOwnOtherResp_r,error=FALSE,include=FALSE,warning=FALSE----------

#generating variable including level of responsResponder_r 
GesamtDaten <- within(GesamtDaten,{
    ownresponsibility_r<-NA
    ownresponsibility_r[treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_7ZplayerX[treatmenttype=="multiple"& GroupRole==3]
    ownresponsibility_r[treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_7YZplayerX[treatmenttype=="computer"& GroupRole==3]
    ownresponsibility_r[treatmenttype=="single" & GroupRole==2]<-100-questionnaire_7YplayerXs[treatmenttype=="single"& GroupRole==2]
    #
    otherresponsibility_r<-NA
    otherresponsibility_r[treatmenttype=="multiple" & GroupRole==3]<-100-questionnaire_7ZplayerY[treatmenttype=="multiple"& GroupRole==3]
    otherresponsibility_r[treatmenttype=="computer" & GroupRole==3]<-100-questionnaire_7YZcomputer[treatmenttype=="computer"& GroupRole==3]
        #
    diff.ownotherresponsibility_r <- ownresponsibility_r - otherresponsibility_r
    })

## ----fig.ownResp---------------------------------------------------------
xx<-GesamtDaten[,c("ownresponsibility","responsBenefPlayer","responsResponder","choiceA","treatmenttype")]
names(xx)[1:3]<-c("y.1","y.2","y.3")
xr<-reshape(xx,direction="long",varying=1:3,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("outcome","payoff co-\\dictator","payoff \\responder")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Responsibility",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(3,1))

## ----ciTableown,results="asis"-------------------------------------------
xx<-rbind(
cbind(dr.test(c("ownresponsibility"),t="computer"),
      dr.test(c("ownresponsibility"),t="single")),
cbind("",dr.test("responsBenefPlayer",t="single")),
cbind(dr.test("responsResponder",t="computer"),
 dr.test("responsResponder",t="single")))
xx<-cbind(c("outcome","payoff co-\\dictator","payoff \\responder"),xx)
colnames(xx)[1]<-"responsibility for\\ldots"
colnames(xx)<-sub("SDT--MDT","SDT--MDT (Hyp. \\\\ref{hyp:resp}.\\\\ref{hyp:SMresp})",colnames(xx))
colnames(xx)<-sub("CDT--MDT","CDT--MDT (Hyp. \\\\ref{hyp:resp}.\\\\ref{hyp:CMresp})",colnames(xx))
xtable(xx)

## ----fig.perc.guilt------------------------------------------------------
ecdfplot(~guilt ,group=factor(treatTrans(treatmenttype),treatSort),
         data=GesamtDaten,auto.key=list(columns=3),xlab="Guilt")

## ----ciTableGuilt,results="asis"-----------------------------------------
xtable(cbind(
    dr.test("guilt",t="computer"),
    dr.test("guilt",t="single")))

## ----choice A dictator,error=FALSE,include=FALSE,warning=FALSE-----------
#------------------------------------------------choice A
#generate dummy variable including only the choice A (all treatments and all ACTIVE players)

#number of dictators (multiple, computer, single)
mySegplot <-function(...) segplot(draw.bands=FALSE,segments.fun = panel.arrows, ends = "both",angle=90,length = 2, 
   unit = "mm",par.strip.text=list(cex=.85),...)

binom.ci <- function (var)  {
    nrchoiceA<-with(GesamtDaten,table(choiceA,factor(treatTrans(treatmenttype))))
    with(binom.test(nrchoiceA[,var]),{out<-rbind(1-c(estimate,conf.int));
        colnames(out)<-c("mean","min","max");
        rownames(out)<-var
        out})}
xx<-rbind(binom.ci("MDT"),binom.ci("CDT"),binom.ci("SDT"))
xx<-within(data.frame(xx),type<-rownames(xx))

## ----fig.choices,fig.height=2--------------------------------------------
mySegplot(factor(type,treatSort) ~ min+max ,centers=mean,data=xx)

## ----decidingalone,error=FALSE,include=FALSE,warning=FALSE---------------
#variable generation [0=Option A/100=Option B](multiple, computer)
GesamtDaten<-within(GesamtDaten,{
    decidingalone<-NA
    decidingalone [treatmenttype=="multiple"& GroupRole!=3]<-questionnaire_1XY[treatmenttype=="multiple"& GroupRole!=3]
    decidingalone [treatmenttype=="computer"& GroupRole==1]<-questionnaire_1X[treatmenttype=="computer"& GroupRole==1]
})

## ----expectationotherdictator,error=FALSE,include=FALSE,warning=FALSE----
#generating variable including level [0=choose A/100=choose B]
GesamtDaten<-within(GesamtDaten,{
    expectationotherdictator<-NA
    expectationotherdictator[treatmenttype=="multiple"& GroupRole!=3]<-questionnaire_2X[treatmenttype=="multiple"& GroupRole!=3]+questionnaire_2Y[treatmenttype=="multiple"& GroupRole!=3]
    expectationotherdictator[treatmenttype=="computer"& GroupRole==1]<-questionnaire_2X[treatmenttype=="computer"& GroupRole==1]
})

## ----influenceOtherDictator,error=FALSE,include=FALSE,warning=FALSE------
#generating variable including level [0=choose A/100=choose B]
GesamtDaten <- within(GesamtDaten,{
  influenceOtherDictator<-NA
  influenceOtherDictator[treatmenttype=="multiple"& GroupRole!=3]<-questionnaire_3XY[treatmenttype=="multiple"& GroupRole!=3]==1
  influenceOtherDictator[treatmenttype=="computer"& GroupRole==1]<-questionnaire_3X[treatmenttype=="computer"& GroupRole==1]==1
})
##
influenceByOther<-round(100*sapply(with(GesamtDaten,by(influenceOtherDictator,list(treatmenttype),mean,na.rm=TRUE)),c),1)

## ----fig.payoffcodictsubResp---------------------------------------------
ecdfplot(~responsBenefPlayer | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten,treatmenttype!='single'),auto.key=list(columns=2),xlab="Responsibility")

## ----ciTablerespsubset_payoffcodict,results="asis"-----------------------
GesamtDaten[,c("responsBenefPlayer","choiceA","treatmenttype", "GroupRole")] %>%
subset(GroupRole==1) %$%
aggregate(responsBenefPlayer ~ treatmenttype + choiceA,FUN=mean)  %>%
reshape(.,direction="wide",v.names="responsBenefPlayer",idvar="treatmenttype",timevar="choiceA") %>%
transform(treatmenttype = revalue(treatmenttype,
                       c("computer"="\\CDT","multiple"="\\MDT"))) %>% 
rename(replace=c("treatmenttype"="Payoff co-dictator",
              "responsBenefPlayer.FALSE"="Option B",
              "responsBenefPlayer.TRUE"="Option A")) %>% 
{ .[2:1,c(1,3,2)] } %>%
xtable()

## ----fig.ownsubResp------------------------------------------------------
ecdfplot(~ownresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("single"="\\SDT", "computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten),auto.key=list(columns=3),xlab="Responsibility")

## ----ciTablerespsubset_outcome,results="asis"----------------------------
GesamtDaten[,c("ownresponsibility","choiceA","treatmenttype", "GroupRole")] %>%
subset(GroupRole==1) %$%
aggregate(ownresponsibility ~ treatmenttype + choiceA,
          FUN=mean) %>%
reshape(.,direction="wide",v.names="ownresponsibility",
        idvar="treatmenttype",timevar="choiceA") %>%
transform(treatmenttype = revalue(treatmenttype,
                       c("single"="\\SDT", "computer"="\\CDT","multiple"="\\MDT"))) %>% 
rename(replace=c("treatmenttype"="Unfair outcome",
              "ownresponsibility.FALSE"="Option B",
              "ownresponsibility.TRUE"="Option A")) %>% 
{ .[3:1,c(1,3,2)] } %>%
xtable()

## ----fig.otherResp-------------------------------------------------------
ecdfplot(~otherresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten,treatmenttype!='single'),auto.key=list(columns=2),xlab="Responsibility")

## ----fig.ownotherResp----------------------------------------------------
ecdfplot(~diff.ownotherresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten,treatmenttype!='single'),auto.key=list(columns=2),xlab="Difference in responsibility")

## ----ciTableother,results="asis"-----------------------------------------
xtable(dr.test("otherresponsibility",t="single",alternative="less"))

## ----decidingalone_r,error=FALSE,include=FALSE,warning=FALSE-------------
#variable generation [0=Option A/100=Option B]
GesamtDaten <- within(GesamtDaten,{
decidingalone_r<-NA
decidingalone_r [treatmenttype=="multiple"& GroupRole==3]<-questionnaire_1Z[treatmenttype=="multiple"& GroupRole==3]
decidingalone_r [treatmenttype=="computer"& GroupRole==3]<-questionnaire_1YZ[treatmenttype=="computer"& GroupRole==3]
})

## ----expectationdictator_r,error=FALSE,include=FALSE,warning=FALSE-------
#generating variable [0=choose A/100=choose B]
GesamtDaten <- within(GesamtDaten,{
    expectationdictator_r<-NA
    expectationdictator_r[treatmenttype=="multiple"& GroupRole==3]<-questionnaire_2Z[treatmenttype=="multiple"& GroupRole==3]
    expectationdictator_r[treatmenttype=="computer"& GroupRole==3]<-questionnaire_2YZ[treatmenttype=="computer"& GroupRole==3]
     expectationdictator_r[treatmenttype=="single"& GroupRole==2]<-questionnaire_2Ys[treatmenttype=="single"& GroupRole==2]
})

## ----decidingalone_pd,error=FALSE,include=FALSE,warning=FALSE------------
#variable generation [0=Option A/100=Option B]
GesamtDaten <- within(GesamtDaten,{
decidingalone_pd<-NA
decidingalone_pd[treatmenttype=="computer"& GroupRole==2]<-questionnaire_1YZ[treatmenttype=="computer"& GroupRole==2]
})

## ----fig.decidingAlone---------------------------------------------------
xx<-GesamtDaten[,c("decidingalone","decidingalone_r","decidingalone_pd","choiceA","treatmenttype")]
names(xx)[1:3]<-c("y.1","y.2","y.3")
xr<-reshape(xx,direction="long",varying=1:3,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("\\dictator","\\responder","\\passivedictator{}")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Propensity to choose Option B",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(3,1))

## ----fig.decideAlone-----------------------------------------------------
ecdfplot(~decidingalone | ifelse(choiceA,"A","B"),group=factor(treatmenttype),data=subset(GesamtDaten,treatmenttype!="single"),auto.key=list(columns=2),xlab="Propensity to choose Option B")

## ----expectationdictator_pd,error=FALSE,include=FALSE,warning=FALSE------
#generating variable including level of expectationdictator_r [0=choose A/100=choose B]
GesamtDaten <- within(GesamtDaten,{
    expectationdictator_pd<-NA
    expectationdictator_pd[treatmenttype=="computer"& GroupRole==2]<-questionnaire_2YZ[treatmenttype=="computer"& GroupRole==2]
})

## ----fig.expectationotherdictatortest------------------------------------
xx<-GesamtDaten[,c("expectationotherdictator","expectationdictator_r","expectationdictator_pd","choiceA","treatmenttype")]
names(xx)[1:3]<-c("y.1","y.2","y.3")
xr<-reshape(xx,direction="long",varying=1:3,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("\\dictator","\\responder","\\passivedictator{}")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Propensity to choose Option B",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(3,1))

## ----fig.expectationotherdictator----------------------------------------
ecdfplot(~expectationotherdictator | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten,treatmenttype!="single"),auto.key=list(columns=2),xlab="Expected probability of choosing Option B")

## ----tab.exp.no.choices,results="asis"-----------------------------------
shortNames <- function(n) {
    n <- sub("computer","\\\\CDT",n)
    n <- sub("multiple","\\\\MDT",n)
    n <- sub("single","\\\\SDT",n)
    n
    }
#
choiceA.tab<-t(round(with(subset(GesamtDaten,Recipient),prop.table(table(treatmenttype,choiceAExp),1))*100,1))
colnames(choiceA.tab)<-paste("recipient",shortNames(colnames(choiceA.tab)))
#
choiceA.tab_pd<-t(round(with(subset(GesamtDaten,IntermediateRecipient),prop.table(table(treatmenttype,choiceAExp_pd),1))*100,1))
colnames(choiceA.tab_pd)<-paste("pass.dict.",shortNames(colnames(choiceA.tab_pd)))
xtable(cbind("exp. no. of A choices"=as.numeric(rownames(choiceA.tab)),choiceA.tab,rbind(choiceA.tab_pd,c(0,NA,NA)))[,1:5],digits=c(0,0,1,1,1,1))
xx<-within(subset(GesamtDaten,Recipient),cAE<-choiceAExp/ifelse(treatmenttype=="multiple",2,1))
SMself.exp.p.value<-summary(with(subset(xx,treatmenttype!="computer"),glm(cAE~treatmenttype,family=binomial)))[["coefficients"]]["treatmenttypesingle","Pr(>|z|)"]
MCself.exp.p.value<-summary(with(subset(xx,treatmenttype!="single"),glm(cAE~treatmenttype,family=binomial)))[["coefficients"]]["treatmenttypemultiple","Pr(>|z|)"]
SCself.exp.p.value<-summary(with(subset(xx,treatmenttype!="multiple"),glm(cAE~treatmenttype,family=binomial)))[["coefficients"]]["treatmenttypesingle","Pr(>|z|)"]

## ----fig.Resp_r----------------------------------------------------------
xx<-GesamtDaten[,c("ownresponsibility_r","otherresponsibility_r","diff.ownotherresponsibility_r","choiceA","treatmenttype")]
names(xx)[1:3]<-c("y.1","y.2","y.3")
xr<-reshape(xx,direction="long",varying=1:3,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("\\dictator","co-\\dictator","difference")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Responsibility",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(3,1))

## ----diff.ownotherresponsibility_pd ,error=FALSE,include=FALSE,warning=FALSE----
#generating variable including level of responsResponder_r 
GesamtDaten <- within(GesamtDaten,{
    ownresponsibility_pd<-NA
    ownresponsibility_pd[treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_7YZplayerX[treatmenttype=="computer"& GroupRole==2]
    #
    otherresponsibility_pd<-NA
    otherresponsibility_pd [treatmenttype=="computer" & GroupRole==2] <- 100-questionnaire_7YZcomputer[treatmenttype=="computer"& GroupRole==2]
    #
    diff.ownotherresponsibility_pd <- ownresponsibility_pd - otherresponsibility_pd
})

## ----fig.Resp_pd---------------------------------------------------------
xx<-GesamtDaten[,c("ownresponsibility_pd","otherresponsibility_pd","diff.ownotherresponsibility_pd","choiceA","treatmenttype")]
names(xx)[1:3]<-c("y.1","y.2","y.3")
xr<-reshape(xx,direction="long",varying=1:3,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("\\dictator","co-\\dictator","difference")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Responsibility",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(3,1))

## ----responsBenefPlayer_r,error=FALSE,include=FALSE,warning=FALSE--------
#generating variable including level of responsBenefPlayer_r
GesamtDaten <- within(GesamtDaten,{
  responsBenefPlayer_r<-NA
  responsBenefPlayer_r[treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_4Z[treatmenttype=="multiple"& GroupRole==3]
  responsBenefPlayer_r[treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_4aYZ[treatmenttype=="computer"& GroupRole==3]
})

## ----responsResponder_r,error=FALSE,include=FALSE,warning=FALSE----------
#generating variable including level of responsResponder_r 
GesamtDaten <- within(GesamtDaten,{
    responsResponder_r<-NA
    responsResponder_r[treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_5Z[treatmenttype=="multiple"& GroupRole==3]
    responsResponder_r[treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_5aYZ[treatmenttype=="computer"& GroupRole==3]
    responsResponder_r[treatmenttype=="single"& GroupRole==2]<-100-questionnaire_5Ys[treatmenttype=="single"& GroupRole==2]
})

## ----respondersJudgement-r-----------------------------------------------
xResp<-subset(GesamtDaten,treatmenttype=="computer" & GroupRole==3)[,c("questionnaire_4aYZ","questionnaire_4bYZ","questionnaire_5aYZ","questionnaire_5bYZ")]
xxResp<-with(xResp,rbind.fill(
     data.frame(who="responsibility of human",whom="...for passive dictator",resp=100-questionnaire_4aYZ),
     data.frame(who="responsibility of computer",whom="...for passive dictator",resp=100-questionnaire_4bYZ),
     data.frame(who="responsibility of human",whom="...for recipient",resp=100-questionnaire_5aYZ),
     data.frame(who="responsibility of computer",whom="...for recipient",resp=100-questionnaire_5bYZ)))
xxResp<-within(xxResp,who<-reorder(who,resp))

## ----fig.respondersJudgement_r-------------------------------------------
ecdfplot(~resp | whom,group=who,data=xxResp,auto.key=list(columns=2),xlab="Responsibility (according to recipients's judgement)")

## ----responspassiv_pd,error=FALSE,include=FALSE,warning=FALSE------------
#generating variable including level of responsResponder_r 
GesamtDaten <- within(GesamtDaten,{
    responspassivdict_pd<-NA
    responspassivdict_pd[treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_4aYZ[treatmenttype=="computer"& GroupRole==2]
    #
    responspassivcomp_pd<-NA
    responspassivcomp_pd[treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_4bYZ[treatmenttype=="computer"& GroupRole==2]
    #
    diff.responspassivdict_pdresponspassivcomp_pd <- responspassivdict_pd - responspassivcomp_pd
})

## ----responsResponder_pd,error=FALSE,include=FALSE,warning=FALSE---------
#generating variable including level of responsResponder_r 
GesamtDaten <- within(GesamtDaten,{
    responsResponder_pd<-NA
    responsResponder_pd[treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_5aYZ[treatmenttype=="computer"& GroupRole==2]

    responsRespondercomp_pd<-NA
    responsRespondercomp_pd[treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_5bYZ[treatmenttype=="computer"& GroupRole==2]
    
    diff.responsResponder_pdresponsRespondercomp_pd <- responsResponder_pd - responsRespondercomp_pd
})

## ----passivedictjudgement-pd---------------------------------------------
xResp<-subset(GesamtDaten,treatmenttype=="computer" & GroupRole==2)[,c("questionnaire_4aYZ","questionnaire_4bYZ","questionnaire_5aYZ","questionnaire_5bYZ")]
xxResp<-with(xResp,rbind.fill(
     data.frame(who="responsibility of human",whom="...for passive dictator",resp=100-questionnaire_4aYZ),
     data.frame(who="responsibility of computer",whom="...for passive dictator",resp=100-questionnaire_4bYZ),
     data.frame(who="responsibility of human",whom="...for recipient",resp=100-questionnaire_5aYZ),
     data.frame(who="responsibility of computer",whom="...for recipient",resp=100-questionnaire_5bYZ)))
xxResp<-within(xxResp,who<-reorder(who,resp))

## ----fig.passivedictjJudgement-pd----------------------------------------
ecdfplot(~resp | whom,group=who,data=xxResp,auto.key=list(columns=2),xlab="Responsibility (according to passive dictator's judgement)")

## ----guilt_r,error=FALSE,include=FALSE,warning=FALSE---------------------
 
#generating variable including level of responsguilt_r 
GesamtDaten <- within(GesamtDaten,{
    guilt_r<-NA
    guilt_r[treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_6Z[treatmenttype=="multiple"& GroupRole==3]
    guilt_r[treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_6YZ[treatmenttype=="computer"& GroupRole==3]
    guilt_r[treatmenttype=="single"& GroupRole==2]<-100-questionnaire_6Ys[treatmenttype=="single"& GroupRole==2]
})

## ----responsguilt_pd ,error=FALSE,include=FALSE,warning=FALSE------------
#generating variable including level of responsguilt_pd  
GesamtDaten <- within(GesamtDaten,{
    responsguilt_pd <-NA
    responsguilt_pd [treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_6YZ[treatmenttype=="computer"& GroupRole==2]
})

## ----fig.Guilt_test------------------------------------------------------
xx<-GesamtDaten[,c("guilt_r","responsguilt_pd","diff.ownotherresponsibility_r","choiceA","treatmenttype")]
names(xx)[1:2]<-c("y.1","y.2")
xr<-reshape(xx,direction="long",varying=1:2,timevar="type")
xr<-within(xr,{
    type<-factor(type)
    levels(type)<-c("\\responder","\\passivedictator")
    q<-rep(NA,length(type))
    ## we have to do ECDF manually, since ecdfplot does not accept NAs
    for(var in levels(type)) 
        for(t in levels(treatmenttype)) {
            cond <- type==var & t==treatmenttype & !is.na(y)
            cond[is.na(cond)]<-FALSE
            if(sum(cond)>2)
                q[cond]<-ecdf(y[cond])(y[cond])
        }
})
xyplot(q~y | type,group=factor(treatTrans(treatmenttype),treatSort),
       data=xr,auto.key=list(columns=3,points=FALSE,lines=TRUE,cex=.7),
       xlab="Guilt",ylab="Empirical CDF",type="s",as.table=TRUE,
       par.strip.text=list(cex=.7),layout=c(2,1))

## ----man.responsBenefDictator,error=FALSE,include=FALSE,warning=FALSE----
#generating variable [1=same level of responsibility/ 2=less responsible / 3=more responsible]
GesamtDaten<-within(GesamtDaten,{
    man.responsBenefPlayer<-NA
    man.responsBenefPlayer[treatmenttype=="multiple"& GroupRole!=3]<-(questionnaire_10X[treatmenttype=="multiple"& GroupRole!=3]+questionnaire_10Y[treatmenttype=="multiple"& GroupRole!=3])
    man.responsBenefPlayer[treatmenttype=="computer"& GroupRole==1]<-questionnaire_10X[treatmenttype=="computer"& GroupRole==1]
    man.responsBenefPlayer<-factor(man.responsBenefPlayer)
    levels(man.responsBenefPlayer)<-c("same","more","less")
    man.responsBenefPlayer<-relevel(man.responsBenefPlayer,"less")
})

## ----man.responsBenefPlayer_r,error=FALSE,include=FALSE,warning=FALSE----
GesamtDaten<-within(GesamtDaten,{
    man.responsBenefPlayer_r<-NA
    man.responsBenefPlayer_r[treatmenttype=="multiple"& GroupRole==3]<-questionnaire_8Z[treatmenttype=="multiple"& GroupRole==3]
    man.responsBenefPlayer_r[treatmenttype=="computer"& GroupRole==3]<-questionnaire_8YZ[treatmenttype=="computer"& GroupRole==3]
    man.responsBenefPlayer_r<-factor(man.responsBenefPlayer_r)
    levels(man.responsBenefPlayer_r)<-c("same","more","less")
    man.responsBenefPlayer_r<-relevel(man.responsBenefPlayer_r,"less")
})

## ----man.responsBenefPlayer_pd,error=FALSE,include=FALSE,warning=FALSE----
#generating variable [1=same level of responsibility/ 2=less responsible / 3=more responsible]
GesamtDaten<-within(GesamtDaten,{
    man.responsBenefPlayer_pd<-NA
    man.responsBenefPlayer_pd[treatmenttype=="computer"& GroupRole==2]<-questionnaire_8YZ[treatmenttype=="computer"& GroupRole==2]
    man.responsBenefPlayer_pd<-factor(man.responsBenefPlayer_pd)
    levels(man.responsBenefPlayer_pd)<-c("same","more","less")
    man.responsBenefPlayer_pd<-relevel(man.responsBenefPlayer_pd,"less")
})

## ----fig.man.responsBenefDictator----------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.responsBenefPlayer")

## ----fig.man.responsBenef_rDictator--------------------------------------
par(mar=c(4,4,1,.1))  
MMosaic("man.responsBenefPlayer_r")

## ----fig.man.responsBenef_pdDictator-------------------------------------
par(mar=c(4,4,1,.1)) 
MMosaic("man.responsBenefPlayer_pd")

## ----man.responsResponder,error=FALSE,include=FALSE,warning=FALSE--------
#generating variable [1=same level of responsibility/ 2=less responsible / 3=more responsible]
GesamtDaten <- within(GesamtDaten,{
    man.responsResponder<-NA
    man.responsResponder[treatmenttype=="multiple"& GroupRole!=3]<-questionnaire_11XY[treatmenttype=="multiple"& GroupRole!=3]
    man.responsResponder[treatmenttype=="computer"& GroupRole==1]<-questionnaire_11X[treatmenttype=="computer"& GroupRole==1]
    man.responsResponder[treatmenttype=="single"& GroupRole==1]<-questionnaire_10Xs[treatmenttype=="single"& GroupRole==1]
    man.responsResponder<-factor(man.responsResponder)
    levels(man.responsResponder)<-c("same","more","less")
    man.responsResponder<-relevel(man.responsResponder,"less")
})

## ----man.responsResponder_r,error=FALSE,include=FALSE,warning=FALSE------
GesamtDaten <- within(GesamtDaten,{
    man.responsResponder_r<-NA
    man.responsResponder_r[treatmenttype=="multiple"& GroupRole==3]<-questionnaire_9Z[treatmenttype=="multiple"& GroupRole==3]
    man.responsResponder_r[treatmenttype=="computer"& GroupRole==3]<-questionnaire_9YZ[treatmenttype=="computer"& GroupRole==3]
    man.responsResponder_r[treatmenttype=="single"& GroupRole==2]<-questionnaire_9Ys[treatmenttype=="single"& GroupRole==2]
    man.responsResponder_r<-factor(man.responsResponder_r)
    levels(man.responsResponder_r)<-c("same","more","less")
    man.responsResponder_r<-relevel(man.responsResponder_r,"less")
})

## ----man.responsResponder_pd,error=FALSE,include=FALSE,warning=FALSE-----
#generating variable including [1=same level of responsibility/ 2=less responsible / 3=more responsible]
GesamtDaten <- within(GesamtDaten,{
    man.responsResponder_pd<-NA
    man.responsResponder_pd[treatmenttype=="computer"& GroupRole==2]<-questionnaire_9YZ[treatmenttype=="computer"& GroupRole==2]
    man.responsResponder_pd<-factor(man.responsResponder_pd)
    levels(man.responsResponder_pd)<-c("same","more","less")
    man.responsResponder_pd<-relevel(man.responsResponder_pd,"less")
})

## ----fig.man.respResponder-----------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.responsResponder")

## ----fig.man.respResponder_r---------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.responsResponder_r")

## ----fig.man.respResponder_pd--------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.responsResponder_pd")

## ----man.guilt,error=FALSE,include=FALSE,warning=FALSE-------------------
#generating variable [1=same level of guilt/ 2=less guilty / 3=more guilty]
GesamtDaten<-within(GesamtDaten,{
    man.guilt<-NA
    man.guilt[treatmenttype=="multiple"& GroupRole!=3]<-questionnaire_12XY[treatmenttype=="multiple"& GroupRole!=3]
    man.guilt[treatmenttype=="computer"& GroupRole==1]<-questionnaire_12X[treatmenttype=="computer"& GroupRole==1]
    man.guilt[treatmenttype=="single"& GroupRole==1]<-questionnaire_12Xs[treatmenttype=="single"& GroupRole==1]
    man.guilt<-factor(man.guilt)
    levels(man.guilt)<-c("same","more","less")
    man.guilt<-relevel(man.guilt,"less")
})

## ----man.guilt_r,error=FALSE,include=FALSE,warning=FALSE-----------------
GesamtDaten<-within(GesamtDaten,{
    man.guilt_r<-NA
    man.guilt_r[treatmenttype=="multiple"& GroupRole==3]<-questionnaire_10Z[treatmenttype=="multiple"& GroupRole==3]
    man.guilt_r[treatmenttype=="computer"& GroupRole==3]<-questionnaire_10YZ[treatmenttype=="computer"& GroupRole==3]
    man.guilt_r[treatmenttype=="single"& GroupRole==2]<-questionnaire_10Ys[treatmenttype=="single"& GroupRole==2]
    man.guilt_r<-factor(man.guilt_r)
    levels(man.guilt_r)<-c("same","more","less")
    man.guilt_r<-relevel(man.guilt_r,"less")
})

## ----man.guilt_pd,error=FALSE,include=FALSE,warning=FALSE----------------
#generating variable including [1=same level of guilt/ 2=less guilty / 3=more guilty]
GesamtDaten<-within(GesamtDaten,{
    man.guilt_pd<-NA
    man.guilt_pd[treatmenttype=="computer"& GroupRole==2]<-questionnaire_10YZ[treatmenttype=="computer"& GroupRole==2]
    man.guilt_pd<-factor(man.guilt_pd)
    levels(man.guilt_pd)<-c("same","more","less")
    man.guilt_pd<-relevel(man.guilt_pd,"less")
})

## ----fig.man.guilt-------------------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.guilt",ylab="Guilt")

## ----fig.man.guilt_r-----------------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.guilt_r",ylab="Guilt")

## ----fig.man.guilt_pd----------------------------------------------------
par(mar=c(4,4,1,.1))
MMosaic("man.guilt_pd",ylab="Guilt")

## ----man.ownresponsibility,error=FALSE,include=FALSE,warning=FALSE-------
#variable including level [0=not responsible at all/100=totally responsible](multiple, computer)
GesamtDaten <- within(GesamtDaten,{
    man.ownresponsibility<-NA
    man.ownresponsibility [treatmenttype=="multiple"& GroupRole!=3]<-100-questionnaire_13XYown[treatmenttype=="multiple"& GroupRole!=3]
    man.ownresponsibility [treatmenttype=="computer"& GroupRole==1]<-100-questionnaire_13Xown[treatmenttype=="computer"& GroupRole==1]
    man.ownresponsibility [treatmenttype=="single"& GroupRole==1]<-100-questionnaire_13Xowns[treatmenttype=="single"& GroupRole==1]
})

## ----fig.ManOwnResponsibility--------------------------------------------
ecdfplot(~man.ownresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT","single"="\\SDT")),
         data=GesamtDaten,auto.key=list(columns=3),xlab="Responsibility")

## ----fig.difManOwnResponsibility-----------------------------------------
ecdfplot(~man.ownresponsibility-ownresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT","single"="\\SDT")),
         data=GesamtDaten,auto.key=list(columns=3),xlab="Change in responsibility")+layer(panel.refline(v=0))

## ----man.otherresponsibility ,error=FALSE,include=FALSE,warning=FALSE----
GesamtDaten <- within(GesamtDaten,{
    man.otherresponsibility<-NA
    man.otherresponsibility [treatmenttype=="multiple"& GroupRole!=3]<-100-questionnaire_13XYother[treatmenttype=="multiple"& GroupRole!=3]
    man.otherresponsibility [treatmenttype=="computer"& GroupRole==1]<-100-questionnaire_13Xother[treatmenttype=="computer"& GroupRole==1]
    man.otherresponsibility [treatmenttype=="single"& GroupRole==1]<-100-questionnaire_13Xothers[treatmenttype=="single"& GroupRole==1]
})

## ----fig.ManOtherResponsibility------------------------------------------
ecdfplot(~man.otherresponsibility,group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT", "single"="\\SDT")),data=GesamtDaten,auto.key=list(columns=3),xlab="Responsibility")

## ----fig.difotherRespManCheck--------------------------------------------
ecdfplot(~man.otherresponsibility-otherresponsibility | ifelse(choiceA,"A","B"),
         group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT")),
         data=subset(GesamtDaten,treatmenttype!='single'),auto.key=list(columns=2),xlab="Difference in personal responsibility")+layer(panel.refline(v=0))

## ----man.ownresponsibility_r,error=FALSE,include=FALSE,warning=FALSE-----
#variable including level [0=not responsible at all/100=totally responsible]
GesamtDaten <- within(GesamtDaten,{
    man.ownresponsibility_r<-NA
    man.ownresponsibility_r [treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_11ZplayerX[treatmenttype=="multiple"& GroupRole==3]
    man.ownresponsibility_r [treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_11YZplayerX[treatmenttype=="computer"& GroupRole==3]
    man.ownresponsibility_r [treatmenttype=="single"& GroupRole==2]<-100-questionnaire_11YplayerXs[treatmenttype=="single"& GroupRole==2]
})

## ----man.ownresponsibility_pd,error=FALSE,include=FALSE,warning=FALSE----
#variable including level of dictators assigned  responsibility [0=not responsible at all/100=totally responsible]
GesamtDaten <- within(GesamtDaten,{
    man.ownresponsibility_pd<-NA
    man.ownresponsibility_pd [treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_11YZplayerX[treatmenttype=="computer"& GroupRole==2]
})

## ----fig.ManOwnResponsibility_r------------------------------------------
ecdfplot(~man.ownresponsibility_r,group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT", "single"="\\SDT")),data=GesamtDaten,auto.key=list(columns=3),xlab="Responsibility")

## ----fig.ManOwnResponsibility_pd-----------------------------------------
ecdfplot(~man.ownresponsibility_pd,group=revalue(factor(treatmenttype),c("computer"="\\CDT")),data=subset(GesamtDaten,treatmenttype=='computer'),auto.key=list(columns=1),xlab="Responsibility")

## ----fig.ManOwnOtherResponsibility_r-------------------------------------
ecdfplot(~man.ownresponsibility_r - ownresponsibility_r,group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT", "single"="\\SDT")),data=GesamtDaten,auto.key=list(columns=3),xlab="Change in responsibility ")+layer(panel.refline(v=0))

## ----fig.ManOwnOtherResponsibility_pd------------------------------------
ecdfplot(~man.ownresponsibility_pd - ownresponsibility_pd,group=revalue(factor(treatmenttype),c("computer"="\\CDT")),data=subset(GesamtDaten,treatmenttype=='computer'),auto.key=list(columns=1),xlab="Change in responsibility")+layer(panel.refline(v=0))

## ----man.otherresponsibility_r,error=FALSE,include=FALSE,warning=FALSE----
GesamtDaten <- within(GesamtDaten,{
    man.otherresponsibility_r<-NA
    man.otherresponsibility_r [treatmenttype=="multiple"& GroupRole==3]<-100-questionnaire_11Zcomputer[treatmenttype=="multiple"& GroupRole==3]
    man.otherresponsibility_r [treatmenttype=="computer"& GroupRole==3]<-100-questionnaire_11YZplayerY[treatmenttype=="computer"& GroupRole==3]
    man.otherresponsibility_r [treatmenttype=="single"& GroupRole==2]<-100-questionnaire_11Ycomputers[treatmenttype=="single"& GroupRole==2]
})

## ----man.otherresponsibility_pd,error=FALSE,include=FALSE,warning=FALSE----
#variable including level of dictators perceived others responsibility [0=not responsible at all/100=totally responsible]
GesamtDaten <- within(GesamtDaten,{
    man.otherresponsibility_pd<-NA
    man.otherresponsibility_pd [treatmenttype=="computer"& GroupRole==2]<-100-questionnaire_11YZplayerY[treatmenttype=="computer"& GroupRole==2]
})

## ----fig.ManOtherResponsibility_r----------------------------------------
ecdfplot(~man.otherresponsibility_r,group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT", "single"="\\SDT")),data=GesamtDaten,auto.key=list(columns=3),xlab="Responsibility")

## ----fig.ManOtherResponsibility_pd---------------------------------------
ecdfplot(~man.otherresponsibility_pd,group=revalue(factor(treatmenttype),c("computer"="\\CDT")),data=subset(GesamtDaten,treatmenttype=='computer'),auto.key=list(columns=1),xlab="Responsibility")

## ----fig.otherRespManCheck_r---------------------------------------------
ecdfplot(~man.otherresponsibility_r-otherresponsibility_r,group=revalue(factor(treatmenttype),c("computer"="\\CDT","multiple"="\\MDT", "single"="\\SDT")),data=subset(GesamtDaten,treatmenttype!='single'),auto.key=list(columns=2),xlab="Difference in personal responsibility")+layer(panel.refline(v=0))

## ----fig.otherRespManCheck_pd--------------------------------------------
ecdfplot(~man.otherresponsibility_pd-otherresponsibility_pd,group=revalue(factor(treatmenttype),c("computer"="\\CDT")),data=subset(GesamtDaten,treatmenttype=='computer'),auto.key=list(columns=1),xlab="Difference in responsibility")+layer(panel.refline(v=0))

## ----effSizeTable,results='asis'-----------------------------------------
effSizeT<-data.frame()
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Forsyth.2002",treat="group size",dep="personal responsibility",n1=122,s1=0,s2=100,e1=12.97,e2=51.84)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Burnette.2008",treat="sucess $\\times$ openness",dep="own responsibility",n1=96,s1=0,s2=100,e1=28.73-24.25,e2=21.83-24.50)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Whyte.1991",treat="group vs.~invidual",dep="personal responsibility",n1=173,s1=0,s2=100,e1=65.5,e2=87.9)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Savitsky.2005",treat="group vs.~invidual",dep="average responsibility",n1=52,s1=1,s2=100,e1=111.0/4,e2=145.9/4)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Gosling.2006",treat="choice",dep="responsibility",n1=51,s1=1,s2=11,e1=6.97,e2=8.85)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Botti.2006",treat="choice",dep="responsibility",n1=96,s1=1,s2=9,e1=6.28,e2=1.94)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Mynatt.1975",treat="group $\\times$ win",dep="responsibility",n1=80,s1=1,s2=7,e1=4.7-4.7,e2=4.67-3.6)))

effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Melo.2016",treat="human/computer",dep="guilt",n1=140,s1=1,s2=7,e1=1.77,e2=1.98)))
effSizeT<-rbind.fill(effSizeT,data.frame(list(auth="Reuben.2010",treat="destruction (period 2)",dep="guilt",n1=55,s1=1,s2=7,e1=2.45,e2=1.75)))
effTab<-within(effSizeT,{
    auth<-sprintf("\\citet{%s}",auth)
    eff100<-abs(e1-e2)/(s2-s1)*100})[,c("auth","treat","dep","n1","eff100")]
xtable(plyr::rename(effTab,replace=c("auth"="Study","treat"="Treatment","dep"="Dependent variable","n1"="$n$","eff100"="\\begin{tabular}{@{}l@{}}Effect size\\\\on a scale\\\\from 0-100\\end{tabular}")),
       digits=c(0,0,0,0,0,1),align=c("c","p{18ex}","p{18ex}","p{22ex}","r","r"))

## ----ciTableownEffsize,results='asis'------------------------------------
print(xtable(rbind(outcome=c(pwr.test(c("ownresponsibility"),t="computer"),
      pwr.test(c("ownresponsibility"),t="single")),
`payoff co-dictator`=c("",pwr.test("responsBenefPlayer",t="single")),
`payoff recipient`=c(pwr.test("responsResponder",t="computer"),
 pwr.test("responsResponder",t="single")))),include.rownames=TRUE)

## ----ciTableGuiltEffsize,results="asis"----------------------------------
xtable(rbind(guilt=c(pwr.test("guilt",t="computer"),
                           pwr.test("guilt",t="single"))))

## ----ciTableotherEffsize,results="asis"----------------------------------
xtable(rbind(responsibility=c(pwr.test("otherresponsibility",t="single",alternative="less"))))

## ----SaveData------------------------------------------------------------
save.image("all.Rdata")

