## ----libraries,cache=FALSE-----------------------------------------------
library(plyr) 
library(lattice)
library(latticeExtra)
library(stringr)
#
ltheme <- canonical.theme(color = FALSE)
ltheme$strip.background$col <- "transparent"
ltheme$par.main.text$font=1
ltheme$par.main.text$cex=1
ltheme$par.sub.text$font=1
lattice.options(default.theme = ltheme)
#
mPal<-brewer.pal(6,"Greys")
mTheme<-custom.theme(fill = mPal)
mTheme$par.main.text$font=1
mTheme$par.main.text$cex=1
mTheme$par.sub.text$font=1
#
library(xtable)
options(xtable.floating=FALSE)
options(xtable.sanitize.text.function=function(x) x)
#
library(runjags)
library(boot)
library(lme4)
if(!interactive()) runjags.options(silent.jags=TRUE, silent.runjags=TRUE)
initJags<-list()
initJags[[1]]<-list(.RNG.seed=1,.RNG.name="base::Mersenne-Twister")
initJags[[2]]<-list(.RNG.seed=2,.RNG.name="base::Super-Duper")
initJags[[3]]<-list(.RNG.seed=3,.RNG.name="base::Wichmann-Hill")
initJags[[4]]<-list(.RNG.seed=4,.RNG.name="lecuyer::RngStream")
#
evidence <- function (odds) {
    o <- 2*abs(log(odds))
    if (o <2) return("no substantial evidence")
    if (o <6) return("positive evidence")
    return("very strong evidence")
}

## ----setup,error=FALSE,include=FALSE,warning=FALSE,cache=TRUE------------
LAG<-function(x) c(NA,x[-length(x)])    
set.seed(123)
bootstrapSize<-200
zTables<-c("subjects","contracts")
library(zTree)
options(zTree.silent=TRUE) # <- less chatty
setwd("Daten")
allFiles<-list.files(".","*.xls",recursive=TRUE)
allFiles<-allFiles[-grep("140708_1343.xls",allFiles)] # <- spurious session, had to be restarted, 
#
treatments<-list("A","B","C")
treat2AT<-list(A=FALSE,B=TRUE,C=TRUE)
treat2Regex<-list(A="^1Run.*",B="^2Run.*",C="^3Run.*")
ATdata<-lapply(treatments,function(t) zTreeTables( grep(treat2Regex[[t]],allFiles,value=TRUE), zTables ))
names(ATdata)<-treatments
setwd("..")
tab2Treat <- function(tables) {
    rbind.fill(lapply(treatments,function(treat) 
        rbind.fill(lapply(tables,function(tab) within(ATdata[[treat]][[tab]],{
            ATpossible<-treat2AT[[treat]]
            condition<-treat})))))}
contractsAll <- tab2Treat(c("contracts"))
subjectsAll <- tab2Treat(c("subjects"))
#drop rubbish (we are only interested in the tables during the last Trading period which is containing all
#tradings done during the first and second trading exp)1
##
ROUNDS<-15
AFV<-(ROUNDS+1)/2*24
cleanP <- function (xx) {
    xx<-xx[with(xx,order(Date,Group,Period,end)),]
    within(xx,{
        ##Now we give every group and subject a unique name (since group 1 on day 1 is not the same
        ##as group 1 on day 2)
        Seller<- sprintf("%s-%d",Date,Seller)
        Buyer<- sprintf("%s-%d",Date,Buyer)
        Group<- as.factor(sprintf("%s-%d",Date,Group))
        ##Now we divide Period by ROUNDS, so what was in the table for Period ROUNDS now becomes a representative for
        ##what where the prices during the first Tradingexperiment during one exp session
        if(exists("price")) {
            Period<-Period/ROUNDS
        } else {
            price<-p
            Period<-(Period+(ROUNDS-1)) %/% ROUNDS
        }
        ## there are few trades which zTree states to take place at second 900. The actual experiment
        ##only lasts 899 seconds. This is only relevant for determining the period, so we fix it there.
        Round <- pmin(ROUNDS,end %/% 60 + 1)
        ##Subtracting the fundamental value from the price for which an asset was traded
        fundamentalValue<-(ROUNDS+1-Round)*24
        overpricing <- price - fundamentalValue
        RD<-overpricing/AFV
        RAD<-abs(overpricing)/AFV
        ## Now we calculate for every price the volatility to which it led. Volatility is defined as
        ## abs(price[t]-price[t-1]), where t is the index of prices during
        ## the same period in the same group
        priceD<-price-LAG(price)
        endD<-end-LAG(end)
        priceD[endD<0]<-NA
        endD[endD<0]<-NA
        if(exists("actualVolume")) {
            priceD <- priceD/ actualVolume
            endD <- endD / actualVolume
        }
        volatility <- abs(priceD)
        aPriceD<-abs(priceD)
    })}
##
#
subjects<-subset(subjectsAll,Treatment==1 & condition %in% c("A","B"))
subjects<-within(subjects,{
    Subject<- sprintf("%s-%d",Date,Subject)
    Group<- as.factor(sprintf("%s-%d",Date,Group))
    RiskSwitch<-apply(subjects[,grep("^Risk[0-9]+$",names(subjects))],1,mean)
    LossSwitch<-apply(subjects[,grep("^LossAversion[0-9]+$",names(subjects))],1,mean)
    Overconf<-ExpectedPercentil/100
})
##
#For the analysis of the paper we only need prices from condition A and B. 
#Condition C is only needed for the Appendix
# get spreadL
cAllSort<-subset(contractsAll,actualVolume>0)
cAllSort<-cAllSort[with(cAllSort,order(Date,Group,Period,Seller,Buyer,end,sell)),]
cAllSort<-within(cAllSort,{spread<-NA;spread[sell==1]<-p[which(sell==1)-1]-p[sell==1]})
if(min(cAllSort$spread,na.rm=TRUE)<0) stop("negative spread")
contractsABC<-cleanP(subset(cAllSort,sell==1))
contracts<-subset(contractsABC,condition %in% c("A","B"))
xx<-merge(contracts,within(subjects,{SRisk<-RiskSwitch;SLoss<-LossSwitch;SOverconf<-Overconf})[,c("Subject","SRisk","SLoss","SOverconf")],by.x="Seller",by.y="Subject",all.x=TRUE)
xx<-merge(xx,within(subjects,{BRisk<-RiskSwitch;BLoss<-LossSwitch;BOverconf<-Overconf})[,c("Subject","BRisk","BLoss","BOverconf")],by.x="Buyer",by.y="Subject",all.x=TRUE)
contracts<-within(xx,{Buyer<-as.factor(Buyer);Seller<-as.factor(Seller)})
##
save(subjects,contracts,contractsABC,file="sp.Rdata")
#
Nsubj<-length(unique(subjects$Subject))/2
Ngroup<-length(levels(as.factor(subjects$Group)))/2

## ----readSbj,results="hide"----------------------------------------------
library(zTree)
setwd("Daten/")
allFiles<-list.files(".","*.sbj",recursive=TRUE)
ja<-zTreeSbj(grep("^2Run./",allFiles,value=TRUE))
setwd("..")

## ----bayesFigCommon------------------------------------------------------
bTrans<-read.csv(text="
var,tex
beta[1],$\\beta_\\ATP$
beta[2],$\\beta_\\NAT$
beta[3],$\\betaR_B$
beta[4],$\\betaR_S$
beta[5],$\\betaL_B$
beta[6],$\\betaL_S$
beta[7],$\\betaO_B$
beta[8],$\\betaO_S$")
dTrans<-read.csv(text="
var,tex
diff[1],$\\beta_\\NAT-\\beta_\\ATP$
diff[2],$\\betaR_S-\\betaR_B$
diff[3],$\\betaL_S-\\betaL_B$
diff[4],$\\betaO_S-\\betaO_B$")
sTrans<-read.csv(text="
var,tex
sd[1],$\\sigma_U$
sd[2],$\\sigma_G$
sd[3],$\\sigma_S$
sd[4],$\\sigma_B$")
#
mySegPlot <- function(df) {
    df<-within(df,{n<-1:length(tex);tex<-reorder(tex,-n)})
    p1<-segplot(tex~ci2.5+ci97.5,panel=function(...) {panel.abline(v=0);panel.refline(h=1:nrow(df));panel.segplot(...)},xlab="$\\beta$",center=ci50,data=df,draw.bands=FALSE,segments.fun=panel.arrows,ends = "both",angle=90,length = 1, unit = "mm")
    p2<-dotplot(tex ~odd ,data=df,panel=function(...) {panel.dotplot(...);panel.refline(v=seq(-5,5,2)/log(10));panel.abline(v=0)},xscale.components=xscale.components.log,xlab="$P(\\beta>0):P(\\beta\\le0)$",scales=list(x=list(log=10),y=list(draw=FALSE)))
    plot(p1,position=c(0,0,.58,1),more=TRUE)
    plot(p2,position=c(.53,0,1,1))
}
#
lmerge<-function(a,b,name) {
    # a is a dataframe, b is a vector
    df<- data.frame(list(var=names(b)))
    df[[name]]<-b
    join(a,df,by="var")
}
#
allSegs <- function(ba,over=TRUE,plot=TRUE,extraBeta=NULL) {
    betaTrans<-rbind(rbind(bTrans[-2,],extraBeta),dTrans[-1,])
    betaTrans[1,1]<-"beta[2]"
    if (over) betaTrans<-rbind(dTrans[1,],rbind(bTrans,extraBeta),dTrans[-1,])
    #
    xx.ci<-as.data.frame(ba$summary$quantiles[,c("2.5%","50%","97.5%")])
    names(xx.ci)<-paste("ci",sub("%","",names(xx.ci)),sep="")
    xx.ci$var<-rownames(xx.ci)
    #
    xx.ci2<-lmerge(lmerge(xx.ci,ba$mcse$sseff,"sseff"),ba$psrf$psrf[,"Point est."],"psrf")
    #
    odds<-ba$summary$statistics[grep("^s(diff|beta)",xx.ci$var,value=TRUE),"Mean"]
    names(odds)<-sub("^s","",names(odds))
    x<-cbind(o1=odds,o2=1-odds)
    minx<-apply(x,1,min)
    minx<-ifelse(minx==0,1,minx)
    odds.df<-adply(x/cbind(minx,minx),1,.id="var",function(x) data.frame(list(odd=x[1]/x[2],odds=paste(signif(x,3),collapse=":"))))
    xx.all<-join(xx.ci2,odds.df)
    #
    beta.df<-join(betaTrans,xx.all)
    sd.df<-join(sTrans,xx.all)
    #
    if(plot) mySegPlot(beta.df)
#    plot(mySegPlot(xx.ci,diffTrans),position=c(.6,.4,1,1),more=TRUE)
#    plot(mySegPlot(xx.ci,sTrans),position=c(.6,0,1,.6))
    list(beta=beta.df,sd=sd.df)
}
cistr<-function(ba,var) paste("$[",paste(sprintf("%.2f",ba$summary$quantiles[var,c("2.5%","97.5%")]),collapse=", "),"]$\\footnote{Eff.~sample size=$",signif(ba$mcse$sseff[var],3),"$, psrf=$",sprintf("%.4f",ba$psrf$psrf[var,1],3),"$.}")
oddstr<-function(x) {
    x<-c(1-x,x)
    if(min(x)>0) x<-x/min(x)
    paste("$",signif(x[2],3),":",signif(x[1],3),"$")
}
##   

## ----estTab--------------------------------------------------------------
estTab <- function(est) {
    cat("\\begin{center}")
    beta.tab<-within(est$beta,ci<-paste("[",paste(signif(ci2.5,3),signif(ci97.5,3),sep=","),"]"))
    beta.tab<-beta.tab[,c("tex","ci50","ci","odds","sseff","psrf")]
    colnames(beta.tab)<-c("","median","CI$_{95}$","odds($\\beta>0$)","sseff","psrf")
    print(xtable(beta.tab,digits=c(0,0,3,0,0,0,4),align=c("c","l","c","c","c","c","c")),include.rownames=FALSE)
    cat("\\endgraf\\bigskip")
    #
    sd.tab<-within(est$sd,ci<-paste("[",paste(signif(ci2.5,3),signif(ci97.5,3),sep=","),"]"))
    sd.tab<-sd.tab[,c("tex","ci50","ci","sseff","psrf")]
    colnames(sd.tab)<-c("","median","CI$_{95}$","sseff","psrf")
    print(xtable(sd.tab,digits=c(0,0,3,0,0,4),align=c("c","l","c","c","c","c")),include.rownames=FALSE)
    cat("\\end{center}\n")
}
#
tradeVolume<-with(aggregate(actualVolume~Group+Period+Round,FUN=sum,data=subset(contracts)),mean(actualVolume))


## ----RiskLossOverconfPlot,cache=TRUE,fig.width=6.0001,fig.height=2.2-----
library(ks)
par(mfcol=c(1,3))
RL.kde<-kde(subjects[,c("RiskSwitch","LossSwitch")],compute.cont=TRUE)
plot(RL.kde,labcex=.5,xlab="Risk preference",ylab="Risk (with losses)")
RO.kde<-kde(subjects[,c("RiskSwitch","Overconf")],compute.cont=TRUE)
plot(RO.kde,labcex=.5,xlab="Risk preference",ylab="Overconfidence")
LO.kde<-kde(subjects[,c("LossSwitch","Overconf")],compute.cont=TRUE)
plot(LO.kde,labcex=.5,xlab="Risk (with losses)",ylab="Overconfidence")

## ----indivPrices,fig.width=5.5,fig.height=3.5----------------------------
gg<-data.frame(with(contracts,{aggregate(end,list(AT=ATpossible,Group=Group,Period=Period),FUN=mean)}))
gg$g <- rank(gg$Group)/2+1/4

with(join(contracts,data.frame(gg)),{
    #GroupAB<-paste(ifelse(ATpossible,"B","A"),sprintf("%02d-%d",g,Period))
    strip=sprintf("%s, period %d",ifelse(ATpossible,"AT possible","AT not possible"),Period)
    xyplot(price ~ end/60 | strip,group=g,type="a",xlab="Time/[minutes]",par.strip.text=list(cex=.8))+xyplot(rep(seq(360,24,-24),each=2)~rep(seq(0,15,1),each=2)[-c(1,32)],t="l",col="red")

    })

## ----loessFig2,fig.width=6,cache=TRUE------------------------------------
myEnd<-data.frame(end=seq(5,900,5))
myEndAT<-rbind(cbind(ATpossible=TRUE,myEnd),cbind(ATpossible=FALSE,myEnd))

myLambda<-function(title,i,span) within(myEnd,{
    title<-title;
    i<-i;
    y<-predict(loess(RD ~ end,data=contracts,span=span),newdata=myEnd)
    time<-7
    condition<-"lambda"
    })

myL2fun <- function(var,title="",i=1,span=.2,midSpan=NA,data) {
    xx<-subset(data,is.finite(contracts[[var]]))
    mp.pred<-predict(loess(as.formula(paste(var,"~ end * ATpossible")),data=xx,span=span),newdata=myEndAT,se=TRUE)
    xx<-within(myEndAT,{
        title<-title;
        y.01<-mp.pred$fit;
        y.00<-mp.pred$fit-mp.pred$se;
        y.02<-mp.pred$fit+mp.pred$se;
        i<-i;
    })
    xx<-within(reshape(xx,direction="long",varying=c("y.00","y.01","y.02")),
           time<-time+3*(ATpossible))
    if(!is.na(midSpan)) xx<-rbind.fill(xx,myLambda(title,i,span=midSpan))
    xx
}

span<-.2
xxABC<-rbind.fill(myL2fun("RD","\\tiny RD: $(P_{\\itk} - \\PF)/\\AF$",1,midSpan=.75,span=span,data=contracts),
           myL2fun("aPriceD","\\tiny $|\\Delta P_{\\itk}|/n_{\\itk}$",2,span=span,data=contracts),
           myL2fun("endD","\\tiny$\\DeltaT_{\\itk}/n_{\\itk}$",3,span=span,data=contracts),
           myL2fun("spread","\\tiny $\\Spread_{\\itk}$",4,span=span,data=contracts))

xxABC<-within(xxABC,title<-reorder(title,i,median))
#
pss<-list(superpose.line=list(lty=c(3,1,3,3,1,3,1),lwd=c(1,1,1,3,3,3,1),col=c(1,1,1,1,1,1,2)))

key = simpleKey(text=c("AT not possible","AT possible","$\\lambda(t)$"),points=FALSE,lines=TRUE,columns=3)
key$lines$lwd<-c(1,3,1)
key$lines$lty<-c(1,1,1)
key$lines$col[3]<-palette()[2]
xyplot(y ~ end/60|title,group=time,data=xxABC,type="l",par.settings=pss,scales="free",key=key,layout=c(4,1),xlab="Time/[minutes]",ylab="",panel=function(...) {panel.xyplot(...);panel.abline(h=0)})

## ----bayesOverpricing,cache=TRUE,dependson=c("setup")--------------------
NumberOfChains=4
# it is good practise to demean our control variables:
demean <- function(names,data) {
    sapply(names,function(n) (data[[n]]<<-(data[[n]]-mean(data[[n]]))/sd(data[[n]])))
    data
}
contractsM<-demean(c("BRisk","SRisk","BLoss","SLoss","BOverconf","SOverconf"),contracts)
contractsMD<-subset(contractsM,!is.na(endD))
#
# here is the “average” overpricing function:
op.loess<-loess(RD ~ end,weights=actualVolume,data=contractsM)
op.pred<-predict(op.loess)
#
#
initJags <-function(chain) {
    chain<-as.numeric(chain)
    list(.RNG.seed=chain,.RNG.name=c("base::Mersenne-Twister","base::Super-Duper","base::Wichmann-Hill","lecuyer::RngStream")[chain])
}
# 
Over.model<-'model {
    for (i in 1:length(overpricing)) {
      overpricing[i]~dnorm(beta0+op.pred[i]*(1+inprod(beta,X[i,]))+
      uG[gid[i]]+uB[bid[i]]+uS[sid[i]],tau[1]*weight[i])
    }
    # group specific effects:
    for (s in 1:max(gid)) {uG[s]   ~ dnorm(0,tau[2])}
    for (s in 1:max(sid)) {uS[s]   ~ dnorm(0,tau[3])}
    for (s in 1:max(bid)) {uB[s]   ~ dnorm(0,tau[4])}
    beta0   ~ dnorm(0,.0001)
    for (i in 1:K) {beta[i] ~ dnorm(0,.0001);sbeta[i]<-ifelse(beta[i]>0,1,0);}
    sbeta[K+1]<-ifelse(beta[5]+beta[6]>0,1,0)
;
    for (i in 1:4) {diff[i] <- beta[2*i]-beta[2*i-1];sdiff[i]<-ifelse(diff[i]>0,1,0)}
    for (i in 1:4) {
       m[i]     ~ dgamma(1,1)
       d[i]     ~ dgamma(1,1)
       tau[i]   ~ dgamma(m[i]^2/d[i]^2,m[i]/d[i]^2)
       sd[i] <- sqrt(1/tau[i])
    }
}'
#
D.model<-'model {
    for (i in 1:length(y)) {
      y[i]~dnorm(inprod(beta,X[i,])+uG[gid[i]]+uB[bid[i]]+uS[sid[i]],tau[1]*weight[i])
    }
    # group specific effects:
    for (s in 1:max(gid)) {uG[s] ~ dnorm(0,tau[2])}
    for (s in 1:max(sid)) {uS[s] ~ dnorm(0,tau[3])}
    for (s in 1:max(bid)) {uB[s] ~ dnorm(0,tau[4])}
    for (i in 1:8) {beta[i] ~ dnorm(0,.0001);sbeta[i]<-ifelse(beta[i]>0,1,0)}
    for (i in 1:4) {diff[i] <- beta[2*i]-beta[2*i-1];sdiff[i]<-ifelse(diff[i]>0,1,0)}
    for (i in 1:4) {
       m[i]     ~ dgamma(1,1)
       d[i]     ~ dgamma(1,1)
       tau[i]   ~ dgamma(m[i]^2/d[i]^2,m[i]/d[i]^2)
       sd[i] <- sqrt(1/tau[i])
    }
}'
over.data=with(contractsM,list(overpricing=RD,X=cbind(as.numeric(ATpossible),1-as.numeric(ATpossible),
    BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),op.pred=op.pred,
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume))
over.data<-within(over.data,K<-ncol(X))
endD.data=with(contractsMD,list(y=endD,X=cbind(1,as.numeric(ATpossible),
    BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume))
priceD.data=with(contractsMD,list(y=aPriceD,X=cbind(1,as.numeric(ATpossible),
    BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume))
over.bayes<-run.jags(model=Over.model,data=over.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)
endD.bayes<-run.jags(model=D.model,data=endD.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)
priceD.bayes<-run.jags(model=D.model,data=priceD.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)

## ----spreadBayes,cache=TRUE,dependson=c("setup")-------------------------
spread.data=with(contractsM,list(y=spread,X=cbind(1,as.numeric(ATpossible),
    BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume))
spread.bayes<-run.jags(model=D.model,data=spread.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)

## ----noWeight,cache=TRUE,dependson=c("setup")----------------------------
overNW.data=with(contractsM,list(overpricing=RD,X=cbind(as.numeric(ATpossible),1-as.numeric(ATpossible),
    BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),op.pred=op.pred,
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume/actualVolume))
overNW.data<-within(overNW.data,K<-ncol(X))
overNW.bayes<-run.jags(model=Over.model,data=overNW.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)

## ----RADBayes,cache=TRUE,dependson=c("setup")----------------------------

overR.data=with(contractsM,list(overpricing=overpricing/fundamentalValue,
    X=cbind(as.numeric(ATpossible),1-as.numeric(ATpossible),
        BRisk,SRisk,BLoss,SLoss,BOverconf,SOverconf),op.pred=op.pred,
    gid=as.numeric(Group),bid=as.numeric(Buyer),sid=as.numeric(Seller),weight=actualVolume))
overR.data<-within(overR.data,K<-ncol(X))
overR.bayes<-run.jags(model=Over.model,data=overR.data,monitor=c("beta","diff","sbeta","sdiff","sd"),inits=initJags,method="parallel",n.chains=NumberOfChains)

## ----saveTheBayes--------------------------------------------------------
save(file="bayes.Rdata",over.bayes,endD.bayes,priceD.bayes,spread.bayes)

## ----bayesCoefPlot,fig.width=6.5,fig.height=3.3,dependson=c("bayesOverpricing","bayesFigCommon")----
over.est<-allSegs(over.bayes)

## ----prettyBayesChange,fig.width=6.5,dependson=c("bayesOverpricing","bayesFigCommon")----
priceD.est<-allSegs(priceD.bayes,FALSE)

## ----prettyBayesSpeed,fig.width=6.5,dependson=c("bayesOverpricing","bayesFigCommon")----
endD.est<-allSegs(endD.bayes,FALSE)

## ----prettyBayesSpread,fig.width=6.5,dependson=c("bayesOverpricing","bayesFigCommon")----
spread.est<-allSegs(spread.bayes,FALSE)

## ----riskTask,results='asis'---------------------------------------------
for(i in 1:10) cat(sprintf("\\AA{%d}{%d}",i,10-i))

## ----zTreeScreenshot,fig.width=2.5,fig.height=1.8------------------------
zTreeEx<-subset(contractsMD,Group=="141126_1201-2" & Period==1 & end<600)
zTreeEx<-zTreeEx[order(zTreeEx[["end"]]),]
par(mar=c(1.5,2,.5,.5),cex=.7,mgp=c(3,.5,0),las=1)
with(zTreeEx,plot(price ~ end,type="o",xlab="",ylab="",ylim=c(0,500),yaxs="i",xlim=c(0,900),xaxs="i",axes=FALSE,cex=.2))
axis(1,at = seq(0,900,100), labels = TRUE, tick = TRUE,mgp=c(3,.2,0))
axis(2,at = seq(0,500,50), labels = TRUE, tick = TRUE)
axis(2,at = seq(0,500,10), labels = FALSE, tick = TRUE,tcl=-.2)
abline(v=seq(0,900,60),col="blue")

## ----loessFig3,fig.width=6,cache=TRUE------------------------------------
myEnd<-data.frame(end=seq(1,60,1))
myEndAT<-rbind(cbind(ATpossible=TRUE,myEnd),cbind(ATpossible=FALSE,myEnd))
xx<-within(contracts,end <- end %% 60)
span<-.2
xxABC<-rbind.fill(myL2fun("RD","\\tiny RD: $(P_{\\itk} - \\PF)/\\AF$",1,span=span,data=xx),
           myL2fun("aPriceD","\\tiny $|\\Delta P_{\\itk}|/n_{\\itk}$",2,span=span,data=xx),
           myL2fun("endD","\\tiny$\\DeltaT_{\\itk}/n_{\\itk}$",3,span=span,data=xx),
           myL2fun("spread","\\tiny $\\Spread_{\\itk}$",4,span=span,data=xx))

xxABC<-within(xxABC,title<-reorder(title,i,median))
#
pss<-list(superpose.line=list(lty=c(3,1,3,3,1,3,1),lwd=c(1,1,1,3,3,3,1),col=c(1,1,1,1,1,1,2)))

key = simpleKey(text=c("AT not possible","AT possible"),points=FALSE,lines=TRUE,columns=2)
key$lines$lwd<-c(1,3)
key$lines$lty<-c(1,1)
xyplot(y ~ end|title,group=time,data=xxABC,type="l",par.settings=pss,scales="free",key=key,layout=c(4,1),xlab="Time/[seconds]",ylab="",panel=function(...) {panel.xyplot(...);panel.abline(h=0)})

## ----over.tab,results='asis'---------------------------------------------
estTab(over.est)

## ----overR.tab,results='asis'--------------------------------------------
overR.est<-allSegs(overR.bayes)
sumRes[["overR.odds"]]<-overR.est[["beta"]][1,"odds"]
estTab(overR.est)

## ----overNW.tab,results='asis'-------------------------------------------
overNW.est<-allSegs(overNW.bayes)
sumRes[["overNW.odds"]]<-overNW.est[["beta"]][1,"odds"]
estTab(overNW.est)

## ----priceD.tab,results='asis'-------------------------------------------
estTab(priceD.est)

## ----endD.tab,results='asis'---------------------------------------------
estTab(endD.est)

## ----spread.tab,results='asis'-------------------------------------------
estTab(spread.est)

## ----figTreatmentC,fig.width=6,cache=TRUE--------------------------------
myEnd<-data.frame(end=seq(0,900,5))
myL2fun <- function(var,title,i) ddply(contractsABC,.(condition),
           function(x,var) within(myEnd,{
               title<-title;
               i<-i;
               y<-predict(loess(as.formula(paste(var,"~ end")),data=x),newdata=myEnd)}),var=var)
xxABC<-rbind.fill(myL2fun("RD","\\tiny RD: $(P_{\\itk} - \\PF)/\\AF$",1),
           myL2fun("aPriceD","\\tiny $|\\Delta P_{\\itk}|/n_{\\itk}$",2),
           myL2fun("endD","\\tiny$\\DeltaT_{\\itk}/n_{\\itk}$",3),
           myL2fun("spread","\\tiny $\\Spread_{\\itk}$",4))
xxABC<-within(xxABC,title<-reorder(title,i,median))
xyplot(y ~ end/60|title,group=condition,data=xxABC,type="l",scales="free",auto.key=list(points=FALSE,lines=TRUE,columns=3),layout=c(4,1),xlab="Time/[minutes]",ylab="")

## ----merData-------------------------------------------------------------
demean <- function(names,data) {
    sapply(names,function(n) (data[[n]]<<-(data[[n]]-mean(data[[n]]))/sd(data[[n]])))
    data
}
contractsM<-demean(c("BRisk","SRisk","BLoss","SLoss","BOverconf","SOverconf"),contracts)
contractsMD<-subset(contractsM,!is.na(endD))
#
bTrans<-read.csv(text="
var,tex
\\(Intercept\\),$\\\\beta_0$
BRisk,$\\\\betaR_B$
SRisk,$\\\\betaR_S$
BLoss,$\\\\betaL_B$
SLoss,$\\\\betaL_S$
BOverconf,$\\\\betaO_B$
SOverconf,$\\\\betaO_S$
ATpossibleTRUE:end2,$\\\\beta_{\\\\ATP \\\\times t^2}$
ATpossibleTRUE:end,$\\\\beta_{\\\\ATP \\\\times t}$
end2,$\\\\beta_{t^2}$
end,$\\\\beta_{t}$
ATpossibleTRUE,$\\\\beta_{\\\\ATP}$")
#
boot2stat <- function(mer.boot) {
 stat<-data.frame(cbind(beta=mer.boot$t0,sigma=apply(mer.boot$t,2,sd)))
 stat<-within(stat,{t<-beta/sigma;p<-sprintf("%.5f",2*pnorm(-abs(t)));t<-sprintf("%.2f",t);sigma<-sprintf("%.3g",sigma);beta<-sprintf("%.3g",beta)})[,c(1,2,4,3)]
 colnames(stat)<-c("$\\beta$","$\\sigma$","$t$","$P>|t|$")
 for(i in 1:nrow(bTrans)) rownames(stat)<-sub(bTrans[i,1],bTrans[i,2],rownames(stat))
 print(xtable(stat),sanitize.rownames.function=function(x) x,sanitize.colnames.function=function(x) x)
}

## ----overMer1------------------------------------------------------------
over.mer<-with(contractsM,{
    end<-end/900
    end2<-end^2
    lmer(RD ~ ATpossible*(end2)+ATpossible*end + BRisk + SRisk + BLoss + SLoss + BOverconf + SOverconf + (1|Group) + (1|Buyer) + (1|Seller),weights=actualVolume)
})
pSimp<-with(contracts,data.frame(cbind(end=rep(seq(0,1,.05),each=2),BRisk=mean(BRisk),SRisk=mean(SRisk),Bloss=mean(BLoss),SLoss=mean(SLoss),BOverconf=mean(BOverconf),SOverconf=mean(SOverconf))))
pSimp<-cbind(one=1,AT=0:1,end2=pSimp$end^2,pSimp)
pSimp<-cbind(pSimp,with(pSimp,cbind(ATend2=AT*end2,ATend=AT*end)))
pSimp<-within(pSimp,{over<-as.matrix(pSimp) %*% cbind(fixef(over.mer));ATl<-factor(ifelse(AT==1,"AT possible","AT not possible"))})

## ----overMerPic----------------------------------------------------------
with(pSimp,xyplot(over ~ end,group=ATl,type="a",xlab="t=Time/[15 minutes]",ylab="RD: $(P_{\\itk} - \\PF)/\\AF$",auto.key=list(corner=c(0.9, 0.1), points = FALSE,lines=TRUE)))

## ----overMerTab,results="asis"-------------------------------------------
over.boot<-bootMer(over.mer, fixef, nsim = bootstrapSize, seed = 123,parallel="multicore",ncpus=NumberOfChains)
boot2stat(over.boot)

## ----priceDMer1----------------------------------------------------------
priceD.mer<-with(contractsMD,{
    lmer(aPriceD ~ ATpossible + BRisk + SRisk + BLoss + SLoss + BOverconf + SOverconf + (1|Group) + (1|Buyer) + (1|Seller))
})
priceD.boot<-bootMer(priceD.mer, fixef, nsim = bootstrapSize, seed = 123,parallel="multicore",ncpus=NumberOfChains)

## ----priceDMerTab,results="asis"-----------------------------------------
boot2stat(priceD.boot)

## ----endDMer1------------------------------------------------------------
endD.mer<-with(contractsMD,{
    lmer(endD ~ ATpossible + BRisk + SRisk + BLoss + SLoss + BOverconf + SOverconf + (1|Group) + (1|Buyer) + (1|Seller))
})
endD.boot<-bootMer(endD.mer, fixef, nsim = bootstrapSize, seed = 123,parallel="multicore",ncpus=NumberOfChains)

## ----endDMerTab,results="asis"-------------------------------------------
boot2stat(endD.boot)

## ----fisherPitman,results='asis'-----------------------------------------
library(coin)
misp<-ddply(contracts,.(Group,condition),function(x) with(x,c(
       meanApriceD=mean(aPriceD,na.rm=TRUE),
       meanEndD=mean(endD,na.rm=TRUE),
       RADw=weighted.mean(abs(RD),actualVolume),
       RAD=mean(abs(RD)),
       RDw=weighted.mean(RD,actualVolume),
       RD=mean(RD)
       )))
fisherPitman<-function(var) {
     ot<-oneway_test(misp[[var]]~factor(condition),data=misp,distribution="exact",alternative='greater')
     Z<-ot@statistic@teststatistic
     N<-length(ot@statistic@xtrans)
     p<-pvalue(ot)
     c(N=N,Z=Z,p=p)
     }
 xx<-rbind(fisherPitman("RAD"),fisherPitman("RADw"),fisherPitman("RD"),fisherPitman("RDw"),fisherPitman("meanApriceD"),fisherPitman("meanEndD"))
rownames(xx)<-c("\\rule{0pt}{3ex}$\\npMean  |P_{\\itk} - \\PF|/\\AF$",
"$ \\npWmean |P_{\\itk} - \\PF|/\\AF$",
"$ \\npMean  (P_{\\itk} - \\PF)/\\AF$",
"$\\npWmean (P_{\\itk} - \\PF)/\\AF$",
"$\\npMean \\Delta P_{i,\\period} $",
"$\\npMean \\DeltaT_{i,\\period} $")
colnames(xx)<-c("$K$","$Z$","$P_{>Z}$") 
xtable(xx,digits=c(0,0,4,4))

## ----riskEarningAsset,results="asis"-------------------------------------
avStocks<-with(subjectsAll,{
    Subject<-sprintf("%s-%d",Date,Subject)
    aggregate(stocks ~ Subject,FUN=mean)})
totProfit<-with(subset(subjectsAll,Period==max(Period)),
     data.frame(list(Subject=sprintf("%s-%d",Date,Subject),totProfit=profitFirstTrading + money)))

initJags <-function(chain) {
    chain<-as.numeric(chain)
    list(.RNG.seed=chain,.RNG.name=c("base::Mersenne-Twister","base::Super-Duper","base::Wichmann-Hill","lecuyer::RngStream")[chain])
}
#
subjects3<-join(join(within(subjects,stocks<-NULL),avStocks),totProfit)
#
simple.model<-'model {
    for (i in 1:length(y)) {
      y[i]~dnorm(beta[1]+beta[2]*x[i]+uG[g[i]],tau[1])
    }
    for (s in 1:max(g)) {uG[s] ~ dnorm(0,tau[2])}

    for (k in 1:2) {
       beta[k]   ~ dnorm(0,.0001)
       sbeta[k]<-ifelse(beta[k]>0,1,0)
    m[k]     ~ dgamma(1,1)
    d[k]     ~ dgamma(1,1)
    tau[k]   ~ dgamma(m[k]^2/d[k]^2,m[k]/d[k]^2)
    sd[k] <- sqrt(1/tau[k])
    }
}'
simpleJags<-function(x,y) {
    jj<-run.jags(simple.model,data=list(y=subjects3[[y]],x=subjects3[[x]],g=as.numeric(subjects3[["Group"]])),monitor=c("sbeta"),inits=initJags,method="parallel",n.chains=NumberOfChains)
    data.frame(list(x=x,y=y,odds=summary(jj)["sbeta[2]","Mean"]))
}
odds3<-ldply(list("RiskSwitch","LossSwitch","Overconf"),function(x) 
    ldply(list("totProfit","stocks"),function(y) 
        simpleJags(x,y)))

oddsx<-within(odds3,{
    x<-revalue(x,c("RiskSwitch"="Risk measure","LossSwitch"="Risk w. losses","Overconf"="Overconfidence"))
    y<-revalue(y,c("totProfit"="Earnings","stocks"="Assets"))
    odds<-sapply(odds,oddstr)
})
#
#
oddsw<-reshape(oddsx,timevar="y",idvar="x",direction="wide")
colnames(oddsw)<-sub("odds.|x","",colnames(oddsw))
print(xtable(oddsw),include.rownames=FALSE)

